package easik.sketch.util.ini;

import java.util.HashMap;

/**
 * Class used to keep information stored under a particular
 * header of an ini file
 * 
 * @author Ian Rutherford 2001
 * @author Kevin Green 2006
 * @since 2006-07-11 Kevin Green
 * @version 2006-07-13 Kevin Green
 */
public class IniFileNode
{
	/**
	 * The name of the header
	 */
	private String _title;
	/**
	 * Hash map of items stored in the node, indexed by the name of the item
	 */
	private HashMap<String, String> _items;

	/**
	 * Default constructor
	 * 
	 * @param title The name of this header
	 */
	public IniFileNode(String title){
		_title = title;
		_items = new HashMap<String, String>();
	}
	
	/**
	 * Returns the name of this header
	 * 
	 * @return The name of this header
	 */
	public String getTitle(){
		return _title;
	}
	
	/**
	 * Returns the number of items in this header
	 * 
	 * @return The number of items in this header
	 */
	public int numItems()
	{
		return _items.size();
	}

	/**
	 * Adds a new item to this header. If it already exists
	 * then it overwrites it with the new value
	 * 
	 * @param name The name of the new item
	 * @param value The value of the new item
	 * @return True if added successfully, false otherwise
	 */
	public boolean addItem(String name, String value)
	{
		if (name.length() == 0)
			return false;

		// make sure the item doesn't already exist
		if(_items.containsKey(name)){
			return changeItem(name, value);
		}

		// add the item at the end of the array
		_items.put(name, value);
		return true;
	}

	/**
	 * Returns the item given by the name
	 * 
	 * @param name The name of the item to be returned
	 * @return The value of the item to be returned, null if it doesn't exist
	 */
	public String getItem(String name)
	{
		// if the item doesn't exist, return an empty string
		if (!_items.containsKey(name))
			return null;
		return _items.get(name);
	}
	
	/**
	 * Returns the HashMap of all items
	 * 
	 * @return The HashMap of all items
	 */
	public HashMap<String, String> getAllItems(){
		return _items;
	}

	/**
	 * Changes the value of an item
	 * 
	 * @param name The name of the item to be changed
	 * @param value The value of the item to be changed
	 * @return True if the change was successfull, false otherwise
	 */
	public boolean changeItem(String name, String value)
	{
		if (!_items.containsKey(name))
			return false;
		_items.put(name, value);
		return true;
	}

	/**
	 * Removes an item from the header
	 * 
	 * @param name The name of the item to be removed
	 * @return True if it was removed successfully, false otherwise
	 */
	public boolean removeItem(String name)
	{
		// if the item doesn't exist, return false
		if (!_items.containsKey(name))
			return false;

		//Remove the item
		_items.remove(name);
		return true;
	}

	/**
	 * Clears all the items from the header
	 */
	public void erase()
	{
		_items.clear();
	}
}
